#!/usr/bin/env bash

. /hive/miners/custom/nock/h-manifest.conf

GPU_CONF="/hive/miners/custom/nock/GPU.conf"

if [[ -f "$GPU_CONF" ]]; then
    GPU_LINE=$(grep -Eo -- '--gpu[= ][0-9,]+' "$GPU_CONF" | head -n 1)
    if [[ -n "$GPU_LINE" ]]; then
        GPU_NUMBERS=$(echo "$GPU_LINE" | grep -oE '[0-9,]+')
        IFS=',' read -r -a active_gpus <<< "$GPU_NUMBERS"
        echo "[>] GPU.conf -> Aktif GPU'lar: ${active_gpus[*]}"
    else
        echo "[!] GPU.conf içinde '--gpu' bulunamadı, tüm GPU'lar aktif sayılacak."
        active_gpus=()
    fi
else
    echo "[!] GPU.conf bulunamadı, tüm GPU'lar aktif sayılacak."
    active_gpus=()
fi

if [[ ${#active_gpus[@]} -eq 0 ]]; then
    gpu_count=$(nvidia-smi -L | wc -l)
    active_gpus=($(seq 0 $((gpu_count - 1))))
    echo "[>] GPU.conf boş, tüm GPU'lar aktif (${active_gpus[*]})"
fi

LOG_FILE="$CUSTOM_LOG_BASENAME.log"
echo "[>] Log dosyası: $LOG_FILE"

if [[ ! -f "$LOG_FILE" ]]; then
    echo "[!] Log dosyası bulunamadı: $LOG_FILE"
    exit 1
fi

accepted_shares=$(grep -c "Heartbeat success" "$LOG_FILE")
rejected_shares=0
total_hashrate_sum=0
declare -A miner_hashrates

for ((miner_index=0; miner_index<${#active_gpus[@]}; miner_index++)); do
    gpu_raw=$(grep "Card-$miner_index" "$LOG_FILE" | grep "p/s" | tail -n 1)
    if [[ -n "$gpu_raw" ]]; then
        hashrate_str=$(echo "$gpu_raw" | grep -o '[0-9.]* p/s' | cut -d' ' -f1)
        hashrate=$(echo "$hashrate_str * 1000000" | bc | cut -d'.' -f1)
    else
        hashrate=0
    fi
    miner_hashrates[$miner_index]=$hashrate
    physical_gpu=${active_gpus[$miner_index]}
    echo "Miner Card-$miner_index  →  Physical GPU $physical_gpu  | Hashrate: ${miner_hashrates[$miner_index]}"
    total_hashrate_sum=$((total_hashrate_sum + hashrate))
done

echo "---------------------------------------"
echo "[>] Toplam Hashrate: $total_hashrate_sum H/s"
echo "---------------------------------------"

if [[ -f "$GPU_STATS_JSON" ]]; then
    readarray -t gpu_stats < <(jq --slurp -r -c '.[] | .busids, .brand, .temp, .fan | join(" ")' "$GPU_STATS_JSON" 2>/dev/null)
    busids=(${gpu_stats[0]})
    brands=(${gpu_stats[1]})
    temps=(${gpu_stats[2]})
    fans=(${gpu_stats[3]})
    gpu_count=${#busids[@]}
else
    echo "[!] GPU_STATS_JSON dosyası bulunamadı!"
    exit 1
fi

hash_arr=()
busid_arr=()
temp_arr=()
fan_arr=()

echo "[DEBUG] Toplam fiziksel GPU sayısı: $gpu_count"
echo "[DEBUG] Bus IDs: ${busids[*]}"
echo "[DEBUG] Brands: ${brands[*]}"

# SADECE NVIDIA GPU'LARI SAY
nvidia_gpu_count=0
nvidia_gpu_indices=()

for ((i=0; i<gpu_count; i++)); do
    # CPU ve onboard GPU'ları atla, sadece NVIDIA GPU'ları al
    if [[ "${brands[i]}" == "nvidia" ]]; then
        nvidia_gpu_indices+=($i)
        ((nvidia_gpu_count++))
        echo "[DEBUG] NVIDIA GPU bulundu: Index $i, Bus ID: ${busids[i]}"
    else
        echo "[DEBUG] Skip ediliyor: Index $i, Brand: ${brands[i]}"
    fi
done

echo "[DEBUG] Toplam NVIDIA GPU sayısı: $nvidia_gpu_count"
echo "[DEBUG] NVIDIA GPU indeksleri: ${nvidia_gpu_indices[*]}"

# NVIDIA GPU'lar için döngü
for nvidia_index in "${!nvidia_gpu_indices[@]}"; do
    physical_index=${nvidia_gpu_indices[$nvidia_index]}
    
    [[ "${busids[physical_index]}" =~ ^([A-Fa-f0-9]+): ]]
    busid_arr+=($((16#${BASH_REMATCH[1]})))
    temp_arr+=(${temps[physical_index]})
    fan_arr+=(${fans[physical_index]})
    
    # Bu NVIDIA GPU aktif mi?
    # NVIDIA index'ini fiziksel index'e çevir
    # Örnek: nvidia_index=0 -> physical_index=1 (onboard atlandı)
    if [[ " ${active_gpus[@]} " =~ " $nvidia_index " ]]; then
        found_index=-1
        for idx in "${!active_gpus[@]}"; do
            if [[ "${active_gpus[$idx]}" == "$nvidia_index" ]]; then
                found_index=$idx
                break
            fi
        done
        if [[ $found_index -ge 0 ]]; then
            gpu_hashrate=${miner_hashrates[$found_index]}
            echo "✓ NVIDIA GPU $nvidia_index (Physical $physical_index) -> Miner Card-$found_index ACTIVE | Hashrate: $gpu_hashrate"
            hash_arr+=($gpu_hashrate)
        else
            echo "✗ NVIDIA GPU $nvidia_index (Physical $physical_index) ACTIVE but no miner data"
            hash_arr+=(0)
        fi
    else
        echo "✗ NVIDIA GPU $nvidia_index (Physical $physical_index) - INACTIVE (not in GPU.conf)"
        hash_arr+=(0)
    fi
done

# Eğer hash_arr boşsa ama NVIDIA GPU varsa, 0 ekle
if [[ ${#hash_arr[@]} -eq 0 ]] && [[ $nvidia_gpu_count -gt 0 ]]; then
    echo "[WARNING] hash_arr boş, NVIDIA GPU'lar için 0 ekleniyor"
    for ((i=0; i<nvidia_gpu_count; i++)); do
        hash_arr+=(0)
    done
fi

hash_json=$(printf '%s\n' "${hash_arr[@]}" | jq -cs '.')
bus_json=$(printf '%s\n' "${busid_arr[@]}" | jq -cs '.')
temp_json=$(printf '%s\n' "${temp_arr[@]}" | jq -cs '.')
fan_json=$(printf '%s\n' "${fan_arr[@]}" | jq -cs '.')

uptime=$(( $(date +%s) - $(stat -c %Y "$CUSTOM_CONFIG_FILENAME") ))

stats=$(jq -nc \
    --argjson hs "$hash_json" \
    --arg ver "$CUSTOM_VERSION" \
    --argjson bus "$bus_json" \
    --argjson temp "$temp_json" \
    --argjson fan "$fan_json" \
    --arg uptime "$uptime" \
    --argjson accepted "$accepted_shares" \
    --argjson rejected "$rejected_shares" \
    '{hs:$hs, hs_units:"hs", algo:"NOCK", ver:$ver, uptime:$uptime, ar:[$accepted,$rejected], bus:$bus, temp:$temp, fan:$fan}')

khs=$(echo "scale=2; $total_hashrate_sum/1000" | bc)

echo ""
echo "==================== DEBUG ===================="
echo "Toplam Hashrate : $khs kH/s"
echo "Accepted Shares : $accepted_shares"
echo "Rejected Shares : $rejected_shares"
echo "Hash Array      : ${hash_arr[*]}"
echo "Output JSON     : $stats"
echo "================================================"